import os
import shutil

import maya.cmds as cmds
import maya.mel as mel

TOOL_NAME = "MaterialMaster"                 
TOOL_MODULE = "MaterialMaster"              
TOOL_ENTRY_COMMAND = "MaterialMaster.run()" 
SHELF_NAME = "Custom"                

ICON_REL_PATH = os.path.join("icons", "materialmaster_icon.png")  


def _get_install_root():

    try:
        this_file = __file__
    except NameError:
        import inspect
        this_file = inspect.getsourcefile(lambda: None)

    return os.path.dirname(os.path.abspath(this_file))


def _copy_file(src, dst_dir):
    if not os.path.exists(dst_dir):
        os.makedirs(dst_dir)
    dst = os.path.join(dst_dir, os.path.basename(src))
    shutil.copy2(src, dst)
    return dst


def _ensure_shelf(shelf_name):

    gShelfTopLevel = mel.eval('$tmpVar=$gShelfTopLevel')

    if cmds.shelfLayout(shelf_name, exists=True):
        return shelf_name

    cmds.shelfLayout(shelf_name, parent=gShelfTopLevel)
    return shelf_name


def _shelf_button_exists(shelf, label):

    children = cmds.shelfLayout(shelf, q=True, ca=True) or []
    for ch in children:
        if cmds.control(ch, q=True, exists=True):
            if cmds.shelfButton(ch, q=True, label=True) == label:
                return True
    return False


# ================== INSTALLER =======================

def install():
    root = _get_install_root()

    tool_py_src = os.path.join(root, TOOL_MODULE + ".py")
    icon_src = os.path.join(root, ICON_REL_PATH)

    if not os.path.exists(tool_py_src):
        cmds.warning("Impossibile trovare lo script: %s" % tool_py_src)
        return

    if not os.path.exists(icon_src):
        cmds.warning("Impossibile trovare l'icona: %s" % icon_src)

    user_script_dir = cmds.internalVar(userScriptDir=True)
    user_icons_dir = cmds.internalVar(userBitmapsDir=True)

    tool_py_dst = _copy_file(tool_py_src, user_script_dir)
    icon_dst = None
    if os.path.exists(icon_src):
        icon_dst = _copy_file(icon_src, user_icons_dir)
        icon_name = os.path.basename(icon_dst)
    else:
        icon_name = "commandButton.png"

    shelf = _ensure_shelf(SHELF_NAME)

    shelf_command = (
        "import importlib\n"
        "import {mod}\n"
        "importlib.reload({mod})\n"
        "{cmd}\n"
    ).format(mod=TOOL_MODULE, cmd=TOOL_ENTRY_COMMAND)

    if _shelf_button_exists(shelf, TOOL_NAME):
        cmds.warning("Esiste già un bottone '%s' nella shelf '%s'. Non verrà creato un duplicato."
                     % (TOOL_NAME, shelf))
    else:
        cmds.shelfButton(
            parent=shelf,
            label="",
            annotation="Created by Nicolò Granese",
            image=icon_name,
            image1=icon_name,
            style="iconOnly",
            command=shelf_command
        )

    cmds.inViewMessage(amg="<hl>{}</hl> installato correttamente sulla shelf <hl>{}</hl>.".format(
        TOOL_NAME, shelf), pos="topCenter", fade=True)


def onMayaDroppedPythonFile(*args):
    install()

if __name__ == "__main__":
    install()
