import maya.cmds as cmds
import re
import json
import os
import random
import difflib


missing_suffix_objects = []
keyword_fields = []
material_fields = []
row_layouts = []
captured_selection = []
captured_roots = []

icon_path = os.path.join(cmds.internalVar(userBitmapsDir=True), "materialmaster_icon.png")


def run():
    MaterialMasterUI()

# ------------------- Material Definer Functions -------------------

def is_probable_material_suffix(suffix):
    if len(suffix) < 5:
        return False
    return difflib.get_close_matches(suffix.lower(), ["mat"], n=1, cutoff=0.6) or suffix.lower().endswith("mat")


def rename_selected_with_suffix(suffix):
    sel = cmds.ls(selection=True)
    if not sel:
        cmds.warning("No object selected.")
        return

    pattern = re.compile(r'^(.*)_[A-Za-z0-9]+Mat_geo$')

    for obj in sel:
        match = pattern.match(obj)
        if match:
            baseName = match.group(1)
        elif obj.endswith("_geo"):
            baseName = obj[:-4]
        else:
            baseName = obj
        newName = baseName + "_" + suffix
        try:
            cmds.rename(obj, newName)
        except:
            cmds.warning(f"Could not rename {obj}")


def remove_suffix_from_selected(*args):

    transforms = _get_captured_or_warn("Remove Material Suffix")
    if not transforms:
        return

    for obj in transforms:
        short = obj.split("|")[-1]
        tokens = short.split("_")
        if len(tokens) >= 2:
            possible_suffix = tokens[-1] if not short.endswith("_geo") else tokens[-2]
            if is_probable_material_suffix(possible_suffix):
                base_name = "_".join(tokens[:-1]) if not short.endswith("_geo") else "_".join(tokens[:-2])
                new_name = base_name + ("_geo" if short.endswith("_geo") else "")
                try:
                    cmds.rename(obj, new_name)  # obj = path completo, new_name = solo nome
                    cmds.warning(f"{short} renamed to {new_name}")
                except:
                    cmds.warning(f"Could not rename {short}")
            else:
                cmds.warning(f"{short}: '{possible_suffix}' doesn't seem a valid material suffix.")
        else:
            cmds.warning(f"{short} doesn't have enough name segments.")


def apply_woodMat(*args): rename_selected_with_suffix("woodMat_geo")
def apply_plasticMat(*args): rename_selected_with_suffix("plasticMat_geo")
def apply_metalMat(*args): rename_selected_with_suffix("metalMat_geo")
def apply_glassMat(*args): rename_selected_with_suffix("glassMat_geo")
def apply_rubberMat(*args): rename_selected_with_suffix("rubberMat_geo")
def apply_fabricMat(*args): rename_selected_with_suffix("fabricMat_geo")


def apply_custom_suffix(*args):
    custom_material = cmds.textField("customSuffixField", query=True, text=True).strip()
    if not custom_material:
        cmds.warning("Insert name material.")
        return
    if not custom_material.endswith("Mat"):
        custom_material += "Mat"
    custom_suffix = custom_material + "_geo"
    rename_selected_with_suffix(custom_suffix)


# ------------------- Keyword Mapping + Smart Rename -------------------
def get_default_keyword_mapping():
    return {
        "wheel": "rubberMat",
        "glass": "glassMat",
        "metal": "metalMat",
        "wood": "woodMat",
        "plastic": "plasticMat",
        "fabric": "fabricMat",
        "tire": "rubberMat",
        "seat": "fabricMat",
    }

mapping_file = os.path.join(cmds.internalVar(userAppDir=True), "material_keyword_mapping.json")


def load_keyword_mapping():
    if os.path.exists(mapping_file):
        with open(mapping_file, "r") as f:
            return json.load(f)
    else:
        return get_default_keyword_mapping()


def save_keyword_mapping(mapping):
    with open(mapping_file, "w") as f:
        json.dump(mapping, f, indent=4)


def smart_rename_from_keywords(*args):

    mapping = load_keyword_mapping()
    transforms = _get_captured_or_warn("Smart Rename")
    if not transforms:
        return

    renamed_count = 0

    for obj in transforms:
        if not cmds.objExists(obj):
            continue

        full = obj
        short = full.split("|")[-1]
        short_lower = short.lower()

        tokens = short.split("_")
        has_geo = bool(tokens) and tokens[-1] == "geo"

        core_tokens = tokens[:-1] if has_geo else tokens[:]

        if core_tokens and is_probable_material_suffix(core_tokens[-1]):
            core_tokens = core_tokens[:-1]

        if core_tokens:
            base_name = "_".join(core_tokens)
        else:
            base_name = short[:-4] if short.endswith("_geo") else short

        base_lower = base_name.lower()

        best_keyword = None
        best_mat = None
        best_len = 0

        for keyword, mat in mapping.items():
            kl = keyword.lower()
            if kl in base_lower and len(kl) > best_len:
                best_keyword = keyword
                best_mat = mat
                best_len = len(kl)

        if not best_mat:
            cmds.warning(f"Smart Rename: '{short}' doesn't match any keyword, skipped.")
            continue

        if base_name:
            new_short = f"{base_name}_{best_mat}_geo"
        else:
            new_short = f"{best_mat}_geo"

        try:
            cmds.rename(full, new_short)
            cmds.warning(f"{short} renamed to {new_short}")
            renamed_count += 1
        except Exception as e:
            cmds.warning(f"Could not rename {short} to {new_short}: {e}")

    if renamed_count > 0:
        cmds.confirmDialog(
            title="Smart Rename Complete",
            message=f"✓ Successfully renamed {renamed_count} object{'s' if renamed_count != 1 else ''}.",
            button=["OK"]
        )

def open_keyword_editor(*args):
    global keyword_fields, material_fields, row_layouts
    mapping = load_keyword_mapping()

    keyword_fields = []
    material_fields = []
    row_layouts = []

    if cmds.window("keywordEditorWin", exists=True):
        cmds.deleteUI("keywordEditorWin")

    win = cmds.window("keywordEditorWin", title="Keyword Mapping Editor", widthHeight=(450, 480),
                      sizeable=True, backgroundColor=[0.11, 0.11, 0.12])

    main_layout = cmds.columnLayout(adjustableColumn=True, rowSpacing=10, columnOffset=["both", 12])

    header_frame = cmds.frameLayout(labelVisible=False, marginHeight=8, marginWidth=0,
                                    backgroundColor=[0.11, 0.11, 0.12])
    cmds.text(label="Keyword Mapping Editor", font="boldLabelFont", height=32,
              backgroundColor=[0.11, 0.11, 0.12], align="left",
              fn="boldLabelFont")
    cmds.setParent('..')

    cmds.separator(height=8, style='none')

    scroll_frame = cmds.frameLayout(labelVisible=False, marginHeight=0, marginWidth=0,
                                    backgroundColor=[0.16, 0.16, 0.18])
    scroll = cmds.scrollLayout(height=300, backgroundColor=[0.16, 0.16, 0.18])
    inner = cmds.columnLayout(adjustableColumn=True, rowSpacing=4, columnOffset=["both", 8])

    for key, value in mapping.items():
        row = cmds.rowLayout(numberOfColumns=3, adjustableColumn=2,
                             columnWidth=[(1, 150), (2, 200), (3, 50)],
                             width=410, parent=inner, backgroundColor=[0.16, 0.16, 0.18])
        kf = cmds.textField(text=key, backgroundColor=[0.22, 0.22, 0.24], height=28)
        mf = cmds.textField(text=value, backgroundColor=[0.22, 0.22, 0.24], height=28)
        cmds.button(label="✕", command=lambda *args, r=row: remove_entry(r),
                    backgroundColor=[0.4, 0.2, 0.2], width=50, height=28)

        keyword_fields.append(kf)
        material_fields.append(mf)
        row_layouts.append(row)
        cmds.setParent("..")

    cmds.setParent(main_layout)
    cmds.separator(height=10, style='none')

    btn_layout = cmds.columnLayout(adjustableColumn=True, rowSpacing=6)
    cmds.button(label="+ Add new entry", command=lambda *args: add_new_entry(inner),
                height=36, backgroundColor=[0.22, 0.22, 0.24])
    cmds.button(label="Save changes", command=lambda *args: save_mapping_from_ui(),
                height=40, backgroundColor=[0.38, 0.65, 0.98])

    cmds.showWindow(win)


def add_new_entry(parent):
    global keyword_fields, material_fields, row_layouts
    row = cmds.rowLayout(numberOfColumns=3, adjustableColumn=2,
                         columnWidth=[(1, 150), (2, 200), (3, 50)],
                         width=410, parent=parent, backgroundColor=[0.16, 0.16, 0.18])
    kf = cmds.textField(placeholderText="Keyword", backgroundColor=[0.22, 0.22, 0.24], height=28)
    mf = cmds.textField(placeholderText="Material", backgroundColor=[0.22, 0.22, 0.24], height=28)
    cmds.button(label="✕", command=lambda *args, r=row: remove_entry(r),
                backgroundColor=[0.4, 0.2, 0.2], width=50, height=28)

    keyword_fields.append(kf)
    material_fields.append(mf)
    row_layouts.append(row)
    cmds.setParent("..")


def remove_entry(row_layout):
    global keyword_fields, material_fields, row_layouts

    if row_layout in row_layouts:
        index = row_layouts.index(row_layout)
        cmds.deleteUI(row_layout)
        del keyword_fields[index]
        del material_fields[index]
        del row_layouts[index]
        cmds.warning(f"Row #{index + 1} removed.")
    else:
        cmds.warning("Layout not found in list (desync).")


def save_mapping_from_ui():
    global keyword_fields, material_fields
    new_mapping = {}
    for kf, mf in zip(keyword_fields, material_fields):
        keyword = cmds.textField(kf, query=True, text=True).strip()
        material = cmds.textField(mf, query=True, text=True).strip()
        if keyword and material:
            new_mapping[keyword] = material
    save_keyword_mapping(new_mapping)
    cmds.warning("Mapping saved succesfully.")
    cmds.deleteUI("keywordEditorWin")

# ------------------- Material Assigner Functions -------------------
predefined_controls = {}
custom_controls = []
custom_controls_container = None


def randomize_color(control):
    r = random.random()
    g = random.random()
    b = random.random()
    cmds.colorSliderGrp(control, edit=True, rgbValue=(r, g, b))


def assign_materials_by_mapping(material_colors):

    transforms = cmds.ls(type="transform")
    for obj in transforms:
        for keyword, color in material_colors.items():
            if keyword in obj:
                material_name = keyword
                if not cmds.objExists(material_name):
                    lambert = cmds.shadingNode("lambert", asShader=True, name=material_name)
                    cmds.warning("Material '{}' created".format(material_name))
                else:
                    lambert = material_name
                cmds.setAttr(lambert + ".color", color[0], color[1], color[2], type="double3")
                sg_name = lambert + "SG"
                if not cmds.objExists(sg_name):
                    sg = cmds.sets(renderable=True, noSurfaceShader=True, empty=True, name=sg_name)
                    cmds.connectAttr(lambert + ".outColor", sg + ".surfaceShader", force=True)
                    cmds.warning("Created shading group '{}' for '{}'".format(sg_name, lambert))
                else:
                    sg = sg_name
                    if not cmds.connectionInfo(sg + ".surfaceShader", isDestination=True):
                        cmds.connectAttr(lambert + ".outColor", sg + ".surfaceShader", force=True)
                        cmds.warning("Reconnected material '{}' to shading group '{}'".format(lambert, sg))
                shapes = cmds.listRelatives(obj, shapes=True, fullPath=True) or []
                if shapes:
                    for shape in shapes:
                        cmds.sets(shape, edit=True, forceElement=sg)
                else:
                    cmds.sets(obj, edit=True, forceElement=sg)
                cmds.warning("Material assigned '{}' to '{}'".format(lambert, obj))
                break


def on_add_custom_row(parent_layout):

    global custom_controls
    row = cmds.rowLayout(numberOfColumns=4, adjustableColumn=2,
                         columnWidth=[(1, 70), (2, 130), (3, 45), (4, 35)],
                         parent=parent_layout, width=290, backgroundColor=[0.16, 0.16, 0.18])
    keyword_field = cmds.textField(placeholderText="Name", backgroundColor=[0.22, 0.22, 0.24], height=26)
    color_ctrl = cmds.colorSliderGrp(rgb=(1.0, 1.0, 1.0),
                                     columnWidth=[(1, 30), (2, 100)], width=130, height=26)

    rnd_btn = cmds.button(label="RND",
                          command=lambda *args: randomize_color(color_ctrl),
                          width=45, height=26, backgroundColor=[0.22, 0.22, 0.24])

    del_btn = cmds.button(label="✕",
                          command=lambda *args: delete_custom_row(row, keyword_field, color_ctrl),
                          width=35, height=26, backgroundColor=[0.4, 0.2, 0.2])

    custom_controls.append((keyword_field, color_ctrl))
    cmds.setParent('..')


def delete_custom_row(row_layout, keyword_field, color_ctrl):

    global custom_controls
    if (keyword_field, color_ctrl) in custom_controls:
        custom_controls.remove((keyword_field, color_ctrl))
    cmds.deleteUI(row_layout)


def on_create_materials(*args):

    global predefined_controls, custom_controls
    material_colors = {}
    for mat, ctrl in predefined_controls.items():
        rgb = cmds.colorSliderGrp(ctrl, query=True, rgbValue=True)
        material_colors[mat] = rgb
    for keyword_field, color_ctrl in custom_controls:
        keyword = cmds.textField(keyword_field, query=True, text=True).strip()
        if keyword:
            rgb = cmds.colorSliderGrp(color_ctrl, query=True, rgbValue=True)
            material_colors[keyword] = rgb
    cmds.warning("Material mapping: " + str(material_colors))
    assign_materials_by_mapping(material_colors)


def extract_material_names_from_geometries(*args):
    global custom_controls, custom_controls_container, captured_selection

    predefined_list = ["woodMat", "plasticMat", "metalMat", "glassMat", "rubberMat", "fabricMat"]

    material_names = []

    transforms = _get_captured_or_warn("Extract Materials")
    if not transforms:
        return

    processed = set()
    for obj in transforms:
        if obj in processed:
            continue
        processed.add(obj)
        if obj.endswith("_geo"):
            tokens = obj.split("_")
            if len(tokens) >= 2:
                material_token = tokens[-2]
                if material_token not in predefined_list:
                    material_names.append(material_token)

    unique_names = sorted(set(material_names))

    if not unique_names:
        cmds.confirmDialog(
            title="Extracted Material Names",
            message="No custom material found.",
            button=["OK"]
        )
        return

    message = "Material names extracted:\n" + "\n".join(unique_names)
    result = cmds.confirmDialog(
        title="Extracted Material Names",
        message=message,
        button=["OK", "Add to Custom"],
        defaultButton="OK",
        cancelButton="OK",
        dismissString="OK"
    )

    if result == "Add to Custom":
        existing_slots = len(custom_controls)
        missing_slots = len(unique_names) - existing_slots

        for _ in range(missing_slots):
            on_add_custom_row(custom_controls_container)

        for i, name in enumerate(unique_names):
            text_field, _ = custom_controls[i]
            cmds.textField(text_field, edit=True, text=name)


def check_created_materials(*args):
    global captured_selection

    missing_materials = []

    transforms = _get_captured_or_warn("Check Created Materials")
    if not transforms:
        return

    processed = set()
    for obj in transforms:
        if obj in processed:
            continue
        processed.add(obj)
        if obj.endswith("_geo"):
            tokens = obj.split("_")
            if len(tokens) >= 2:
                material_token = tokens[-2]
                if not cmds.objExists(material_token):
                    missing_materials.append(material_token)
    unique_missing = sorted(set(missing_materials))
    if unique_missing:
        message = "The following materials were not created:\n" + "\n".join(unique_missing)
    else:
        message = "✓ All extracted materials have been created."
    cmds.confirmDialog(title="Check Created Materials", message=message, button=["OK"])


def capture_selection(*args):

    global captured_selection, captured_roots
    sel = cmds.ls(selection=True, long=True) or []

    if not sel:
        cmds.warning("No objects selected. Please select geometries or groups.")
        captured_roots = []
        captured_selection = []
        return

    captured_roots = sel

    geos = []
    for root in captured_roots:
        if not cmds.objExists(root):
            continue
        expanded = [root]
        descendants = cmds.listRelatives(root, allDescendents=True,
                                         type="transform", fullPath=True) or []
        expanded.extend(descendants)
        for obj in expanded:
            shapes = cmds.listRelatives(obj, shapes=True, fullPath=True) or []
            if any(cmds.nodeType(shape) == "mesh" for shape in shapes):
                geos.append(obj)

    geos = sorted(set(geos))
    captured_selection = geos

    if captured_selection:
        count = len(captured_selection)
        cmds.confirmDialog(
            title="Selection Captured",
            message=f"✓ Captured {count} geometr{'y' if count == 1 else 'ies'}.\n\n"
                    f"Utility tools will now work on this selection.",
            button=["OK"]
        )
        cmds.warning(f"Captured {count} geometries for utility tools.")
    else:
        cmds.warning("No mesh geometries found in selection.")
        captured_selection = []


def _get_captured_or_warn(action_name="this tool"):

    global captured_roots

    if not captured_roots:
        cmds.warning("No selection captured. Please use 'Capture Selection' first.")
        cmds.confirmDialog(
            title="No Selection",
            message="Please capture a selection first using the 'Capture Selection' button.",
            button=["OK"]
        )
        return []

    geos = []
    for root in captured_roots:
        if not cmds.objExists(root):
            continue
        expanded = [root]
        descendants = cmds.listRelatives(root, allDescendents=True,
                                         type="transform", fullPath=True) or []
        expanded.extend(descendants)
        for obj in expanded:
            shapes = cmds.listRelatives(obj, shapes=True, fullPath=True) or []
            if any(cmds.nodeType(shape) == "mesh" for shape in shapes):
                geos.append(obj)

    geos = sorted(set(geos))

    if not geos:
        cmds.warning("Captured roots no longer contain any mesh geometries.")
        cmds.confirmDialog(
            title="Empty Selection",
            message="Captured roots do not contain any mesh geometries anymore.\n\n"
                    "Please capture a new selection.",
            button=["OK"]
        )
        return []

    cmds.warning(f"{action_name}: using captured selection ({len(geos)} geometries).")
    return geos


def check_missing_suffix(*args):

    global missing_suffix_objects
    missing_suffix_objects = []

    transforms = _get_captured_or_warn("Check Missing Suffix")
    if not transforms:
        return

    for obj in transforms:
        short = obj.split("|")[-1]
        tokens = short.split("_")
        if len(tokens) >= 2:
            possible_suffix = tokens[-2] if short.endswith("_geo") else tokens[-1]
            if not is_probable_material_suffix(possible_suffix):
                missing_suffix_objects.append(obj)
        else:
            missing_suffix_objects.append(obj)

    if missing_suffix_objects:
        short_names = [o.split("|")[-1] for o in missing_suffix_objects]
        msg = (
            f"Geometries with missing or incorrect suffix "
            f"({len(missing_suffix_objects)} found):\n\n" + "\n".join(short_names)
        )
    else:
        msg = "✓ All geometries in the selection have a correct suffix."

    cmds.confirmDialog(title="Check Missing Suffix", message=msg, button=["OK"])


def isolate_missing_suffix(*args):

    global missing_suffix_objects

    if not missing_suffix_objects:
        msg = "There is no geometry to isolate. Press 'Check Missing Suffix' first."
        cmds.confirmDialog(title="Check Missing Suffix", message=msg, button=["OK"])
        return

    existing = [obj for obj in missing_suffix_objects if cmds.objExists(obj)]
    if not existing:
        cmds.warning("Stored list is outdated. Run 'Check Missing Suffix' again.")
        cmds.confirmDialog(
            title="Outdated List",
            message="No valid geometries found.\n\nPlease run 'Check Missing Suffix' again.",
            button=["OK"]
        )
        missing_suffix_objects = []
        return

    cmds.select(existing, replace=True)

    panel = cmds.getPanel(withFocus=True)
    if not cmds.getPanel(typeOf=panel) == "modelPanel":
        panels = cmds.getPanel(type="modelPanel")
        panel = panels[0] if panels else None

    if not panel:
        cmds.warning("No model panel found to isolate.")
        return

    cmds.isolateSelect(panel, state=False)
    cmds.select(existing, replace=True)
    cmds.isolateSelect(panel, state=True)
    cmds.isolateSelect(panel, addSelected=True)

    cmds.warning("Isolate active on objects with missing suffix.")


def auto_assign_materials(*args):

    transforms = _get_captured_or_warn("Smart Assign")
    if not transforms:
        return

    material_to_objs = {}
    for obj in transforms:
        if not cmds.objExists(obj):
            continue

        short = obj.split("|")[-1]
        if not short.endswith("_geo"):
            continue

        tokens = short.split("_")
        if len(tokens) < 2:
            continue

        mat_name = tokens[-2]

        if not is_probable_material_suffix(mat_name):
            continue

        material_to_objs.setdefault(mat_name, []).append(obj)

    if not material_to_objs:
        cmds.confirmDialog(
            title="Smart Assign",
            message=(
                "No valid material suffix found in captured selection.\n\n"
                "Expected names like '*_woodMat_geo', '*_metalMat_geo', etc.\n"
                "Smart Assign will not create any material."
            ),
            button=["OK"]
        )
        cmds.warning("Smart Assign: no valid '*Mat_geo' suffix found. Aborting without creating materials.")
        return

    for mat_name, objs in material_to_objs.items():
        if not cmds.objExists(mat_name):
            lambert = cmds.shadingNode("lambert", asShader=True, name=mat_name)
            cmds.warning("Material created '{}'".format(mat_name))
        else:
            lambert = mat_name

        r, g, b = random.random(), random.random(), random.random()
        cmds.setAttr(lambert + ".color", r, g, b, type="double3")

        sg_name = mat_name + "SG"
        if not cmds.objExists(sg_name):
            sg = cmds.sets(renderable=True, noSurfaceShader=True, empty=True, name=sg_name)
            cmds.connectAttr(lambert + ".outColor", sg + ".surfaceShader", force=True)
        else:
            sg = sg_name
            if not cmds.connectionInfo(sg + ".surfaceShader", isDestination=True):
                cmds.connectAttr(lambert + ".outColor", sg + ".surfaceShader", force=True)

        for obj in objs:
            if not cmds.objExists(obj):
                continue
            shapes = cmds.listRelatives(obj, shapes=True, fullPath=True) or []
            if shapes:
                for shape in shapes:
                    cmds.sets(shape, edit=True, forceElement=sg)
            else:
                cmds.sets(obj, edit=True, forceElement=sg)


def remove_all_materials(*args):

    transforms = _get_captured_or_warn("Remove Materials")
    if not transforms:
        return

    transforms = [obj for obj in transforms if cmds.objExists(obj)]
    if not transforms:
        cmds.confirmDialog(
            title="Remove Materials",
            message="No valid geometries found in captured selection.",
            button=["OK"]
        )
        return

    material_names = set()
    for obj in transforms:
        if obj.endswith("_geo"):
            tokens = obj.split("_")
            if len(tokens) >= 2:
                material_names.add(tokens[-2])

    default_sg = "initialShadingGroup"
    if not cmds.objExists(default_sg):
        cmds.warning("initialShadingGroup not found!")
        return

    for obj in transforms:
        try:
            shapes = cmds.listRelatives(obj, shapes=True, fullPath=True) or []
            if shapes:
                for shape in shapes:
                    cmds.sets(shape, edit=True, forceElement=default_sg)
            else:
                cmds.sets(obj, edit=True, forceElement=default_sg)
        except:
            pass

    cmds.warning("Reassigned lambert1 to captured geometries.")

    deleted_count = 0
    for mat_name in sorted(material_names):
        if cmds.objExists(mat_name):
            try:
                sg_name = mat_name + "SG"
                if cmds.objExists(sg_name):
                    cmds.delete(sg_name)
                cmds.delete(mat_name)
                deleted_count += 1
            except Exception as e:
                cmds.warning("Impossible to delete {}: {}".format(mat_name, e))

    if cmds.objExists("debugMat"):
        try:
            if cmds.objExists("debugMatSG"):
                cmds.delete("debugMatSG")
            cmds.delete("debugMat")
            deleted_count += 1
        except Exception:
            pass

# ------------------- Combined UI -------------------
def MaterialMasterUI():

    global predefined_controls, custom_controls, custom_controls_container

    if cmds.window("materialToolWindow", exists=True):
        cmds.deleteUI("materialToolWindow")

    window = cmds.window("materialToolWindow", title="Material Tools",
                         width=700, height=720,
                         sizeable=True, backgroundColor=[0.11, 0.11, 0.12])

    mainLayout = cmds.rowLayout(numberOfColumns=2,
                                columnWidth=[(1, 338), (2, 338)],
                                backgroundColor=[0.11, 0.11, 0.12])

    leftColumn = cmds.columnLayout(adjustableColumn=True, rowSpacing=8, width=330)

    # ============ CAPTURE SELECTION SECTION (ABOVE ALL) ============
    captureFrame = cmds.frameLayout(labelVisible=False, marginHeight=8, marginWidth=8,
                                    backgroundColor=[0.16, 0.16, 0.18],
                                    borderVisible=False, parent=leftColumn)
    captureLayout = cmds.columnLayout(adjustableColumn=True, rowSpacing=6, width=310,
                                      backgroundColor=[0.16, 0.16, 0.18])

    cmds.separator(height=2, style='none')

    cmds.text(label="To start, select your main group or your desired geos",
              align="center", font="smallPlainLabelFont", height=18,
              backgroundColor=[0.16, 0.16, 0.18])
    cmds.separator(height=4, style='none')

    cmds.button(label="📌  Capture Selection", command=capture_selection,
                height=44, backgroundColor=[0.38, 0.65, 0.98],
                annotation="Capture selection - ALL TOOLS will work on this")

    cmds.separator(height=2, style='none')
    cmds.setParent(leftColumn)
    cmds.separator(height=8, style='none')

    # ---------------- Material Definer Section ----------------
    definerFrame = cmds.frameLayout(label="Material Definer", collapsable=True, collapse=False,
                                    marginHeight=10, marginWidth=8,
                                    backgroundColor=[0.16, 0.16, 0.18],
                                    borderVisible=False, font="boldLabelFont",
                                    labelAlign="center", parent=leftColumn)
    definerLayout = cmds.columnLayout(adjustableColumn=True, rowSpacing=4, width=310,
                                      backgroundColor=[0.16, 0.16, 0.18])

    cmds.separator(height=4, style='none')

    cmds.text(label="Predefined Materials", align="left", font="smallBoldLabelFont",
              height=20, backgroundColor=[0.16, 0.16, 0.18])
    cmds.separator(height=4, style='none')

    materials = [
        ("woodMat", apply_woodMat),
        ("plasticMat", apply_plasticMat),
        ("metalMat", apply_metalMat),
        ("glassMat", apply_glassMat),
        ("rubberMat", apply_rubberMat),
        ("fabricMat", apply_fabricMat)
    ]

    for i in range(0, len(materials), 2):
        row = cmds.rowLayout(numberOfColumns=2, columnWidth=[(1, 150), (2, 150)],
                             adjustableColumn=2, parent=definerLayout,
                             backgroundColor=[0.16, 0.16, 0.18])
        cmds.button(label=materials[i][0], command=materials[i][1],
                    height=32, backgroundColor=[0.22, 0.22, 0.24], width=150)
        if i + 1 < len(materials):
            cmds.button(label=materials[i+1][0], command=materials[i+1][1],
                        height=32, backgroundColor=[0.22, 0.22, 0.24], width=150)
        cmds.setParent('..')

    cmds.separator(height=8, style='none')

    cmds.text(label="Custom Material", align="left", font="smallBoldLabelFont",
              height=20, backgroundColor=[0.16, 0.16, 0.18])
    cmds.separator(height=4, style='none')

    cmds.textField("customSuffixField", placeholderText="Insert material name",
                   backgroundColor=[0.22, 0.22, 0.24], height=30)
    cmds.separator(height=4, style='none')
    cmds.button(label="Apply Custom Material", command=apply_custom_suffix,
                height=34, backgroundColor=[0.22, 0.22, 0.24])

    cmds.setParent(leftColumn)
    cmds.separator(height=8, style='none')

    # ---------------- Smart Material Section ----------------
    smartFrame = cmds.frameLayout(label="Smart Material Definer", collapsable=True, collapse=False,
                                  marginHeight=10, marginWidth=8,
                                  backgroundColor=[0.16, 0.16, 0.18],
                                  borderVisible=False, font="boldLabelFont",
                                  labelAlign="center", parent=leftColumn)
    smartLayout = cmds.columnLayout(adjustableColumn=True, rowSpacing=4, width=310,
                                    backgroundColor=[0.16, 0.16, 0.18])

    cmds.separator(height=4, style='none')

    cmds.text(label="Utility Tools", align="left", font="smallBoldLabelFont",
              height=20, backgroundColor=[0.16, 0.16, 0.18])
    cmds.separator(height=4, style='none')

    utility_buttons = [
        ("Check Missing Suffix", check_missing_suffix),
        ("Isolate Missing Suffix", isolate_missing_suffix),
        ("Edit Keyword Mapping", open_keyword_editor),
        ("Remove Material Suffix", remove_suffix_from_selected)
    ]

    for label, cmd in utility_buttons:
        cmds.button(label=label, command=cmd, height=32, backgroundColor=[0.22, 0.22, 0.24])

    cmds.separator(height=8, style='none')

    cmds.button(label="🔍  Smart Rename", command=smart_rename_from_keywords,
                height=40, backgroundColor=[0.38, 0.65, 0.98],
                annotation="Automatically rename based on keywords")

    cmds.setParent(mainLayout)

    rightColumn = cmds.columnLayout(adjustableColumn=True, rowSpacing=8, width=330)

    assignerFrame = cmds.frameLayout(label="Material Assigner", collapsable=True, collapse=False,
                                     marginHeight=10, marginWidth=8,
                                     backgroundColor=[0.16, 0.16, 0.18],
                                     borderVisible=False, font="boldLabelFont",
                                     labelAlign="center", parent=rightColumn)
    assignerLayout = cmds.columnLayout(adjustableColumn=True, rowSpacing=4, width=310,
                                       backgroundColor=[0.16, 0.16, 0.18])

    cmds.separator(height=4, style='none')

    cmds.text(label="Predefined Materials", align="left", font="smallBoldLabelFont",
              height=20, backgroundColor=[0.16, 0.16, 0.18])
    cmds.separator(height=4, style='none')

    predefined_materials = {
        "woodMat": (0.08, 0.06, 0),
        "plasticMat": (0, 0.02, 0.09),
        "metalMat": (0.07, 0.07, 0.07),
        "glassMat": (0.65, 0.9, 1.2),
        "rubberMat": (0.02, 0.02, 0.02),
        "fabricMat": (0.75, 0.65, 0.45)
    }

    predefined_controls = {}
    for mat, color in predefined_materials.items():
        row = cmds.rowLayout(numberOfColumns=3, adjustableColumn=2,
                             columnWidth=[(1, 85), (2, 170), (3, 45)],
                             parent=assignerLayout, height=28,
                             backgroundColor=[0.16, 0.16, 0.18])
        cmds.text(label=mat, width=85, align="center", backgroundColor=[0.22, 0.22, 0.24])
        color_ctrl = cmds.colorSliderGrp(rgb=color, columnWidth=[(1, 35), (2, 135)],
                                         width=170, height=26)
        predefined_controls[mat] = color_ctrl

        cmds.button(label="RND",
                    command=lambda *args, ctrl=color_ctrl: randomize_color(ctrl),
                    width=45, height=26, backgroundColor=[0.22, 0.22, 0.24])
        cmds.setParent('..')

    cmds.separator(height=8, style='none')

    cmds.text(label="Custom Materials", align="left", font="smallBoldLabelFont",
              height=20, backgroundColor=[0.16, 0.16, 0.18])
    cmds.separator(height=4, style='none')

    custom_controls_container = cmds.columnLayout(adjustableColumn=True,
                                                  parent=assignerLayout, width=310,
                                                  backgroundColor=[0.16, 0.16, 0.18])
    custom_controls = []

    for i in range(2):
        on_add_custom_row(custom_controls_container)

    cmds.setParent(assignerLayout)
    cmds.separator(height=4, style='none')
    cmds.button(label="+ Add Custom Row",
                command=lambda *args: on_add_custom_row(custom_controls_container),
                height=28, backgroundColor=[0.22, 0.22, 0.24])

    cmds.separator(height=8, style='none')

    btn_row = cmds.rowLayout(numberOfColumns=2, columnWidth=[(1, 150), (2, 150)],
                             adjustableColumn=2, parent=assignerLayout,
                             backgroundColor=[0.16, 0.16, 0.18])
    cmds.button(label="Extract Materials", command=extract_material_names_from_geometries,
                height=32, backgroundColor=[0.22, 0.22, 0.24])
    cmds.button(label="Check Created", command=check_created_materials,
                height=32, backgroundColor=[0.22, 0.22, 0.24])
    cmds.setParent(assignerLayout)

    cmds.separator(height=8, style='none')

    cmds.button(label="Create Materials", command=on_create_materials,
                height=36, backgroundColor=[0.22, 0.22, 0.24])
    cmds.separator(height=4, style='none')
    cmds.button(label="⚡  Smart Assign", command=auto_assign_materials,
                height=40, backgroundColor=[0.38, 0.65, 0.98],
                annotation="Automatically assign random materials")
    cmds.separator(height=4, style='none')
    cmds.button(label="🗑  Remove Materials", command=remove_all_materials,
                height=36, backgroundColor=[0.6, 0.25, 0.25],
                annotation="Delete all materials and reassign lambert1")

    cmds.showWindow(window)
